/************************************************************************
	main.c

    LED Light String Display
    Copyright (C) 2009 Simon Inns

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Email: simon.inns@gmail.com

************************************************************************/

// Global includes
#include <htc.h>

// Local includes
#include "hardware.h"

// PIC 18F2550 fuse configuration:
// Config word 1 (Oscillator configuration)
__CONFIG(1, USBPLL & IESODIS & FCMDIS & HSPLL & CPUDIV1 & PLLDIV5);
// Config word 2
__CONFIG(2, VREGEN & PWRTDIS & BORDIS & WDTDIS);
// Config word 3
__CONFIG(3, PBDIGITAL & LPT1DIS & MCLREN);
// Config word 4
__CONFIG(4, XINSTDIS & STVRDIS & LVPDIS & ICPORTDIS & DEBUGDIS);
// Config word 5, 6 and 7 (protection configuration)
__CONFIG(5, UNPROTECT);
__CONFIG(6, UNPROTECT);
__CONFIG(7, UNPROTECT);

// Define globals for interrupt handling
unsigned char globalPWMcounter = 0;

// High priority interrupt procedure
void interrupt hpHandler(void)
{
	// Nothing to do here at the moment
}

// Low priority interrupt procedure
void interrupt low_priority lpHandler(void)
{
	// Is this timer0 interrupting?
	if (TMR0IF)
	{
		// Increment the PWM counter
		globalPWMcounter++;
		if (globalPWMcounter == 63) globalPWMcounter = 0;

		// Control the LED fading

		// fadeOnSpeed and fadeOffSpeed:
		// These control the speed at which the LED reaches the desired brightness (or dimness).
		// If the fadeSpeed is 0 the LED goes straight to full brightness
		// If the fadeSpeed is 1 the LED takes 1*64*208uS to get to full brightness (1/75th of a second)
		// If the fadeSpeed is 75 the LED takes approx. 1 second to get to full brightness, etc.

		unsigned char ledNumber;

		for (ledNumber = 0; ledNumber < 7; ledNumber++)
		{
			// Increment the fadeCounter
			led[ledNumber].fadeCounter++;

			// Fading on?
			if (led[ledNumber].currentBrightness < led[ledNumber].brightness)
			{
				if (led[ledNumber].fadeOnSpeed == 0)
				{
					led[ledNumber].currentBrightness = led[ledNumber].brightness;
					led[ledNumber].fadeCounter = 0;
				}
				else
				{
					if (led[ledNumber].fadeCounter >= led[ledNumber].fadeOnSpeed)
					{
						led[ledNumber].currentBrightness++;
						led[ledNumber].fadeCounter = 0;
					}
				}
			}

			// Fading off?
			if (led[ledNumber].currentBrightness >= led[ledNumber].brightness)
			{
				if (led[ledNumber].fadeOffSpeed == 0)
				{
					led[ledNumber].currentBrightness = led[ledNumber].brightness;
					led[ledNumber].fadeCounter = 0;
				}
				else
				{
					if (led[ledNumber].fadeCounter == led[ledNumber].fadeOffSpeed)
					{
						led[ledNumber].currentBrightness--;
						led[ledNumber].fadeCounter = 0;
					}
				}
			}

			// At the correct brightness?
			if (led[ledNumber].brightness == led[ledNumber].currentBrightness)
				led[ledNumber].fadeCounter = 0;
		}

		// Update the physical LED pins according to the PWM counter position
		if (led[0].currentBrightness > globalPWMcounter) LED0 = 1; else LED0 = 0;
		if (led[1].currentBrightness > globalPWMcounter) LED1 = 1; else LED1 = 0;
		if (led[2].currentBrightness > globalPWMcounter) LED2 = 1; else LED2 = 0;
		if (led[3].currentBrightness > globalPWMcounter) LED3 = 1; else LED3 = 0;
		if (led[4].currentBrightness > globalPWMcounter) LED4 = 1; else LED4 = 0;
		if (led[5].currentBrightness > globalPWMcounter) LED5 = 1; else LED5 = 0;
		if (led[6].currentBrightness > globalPWMcounter) LED6 = 1; else LED6 = 0;

		// Get ready for the next interrupt
		TMR0H = 0xFB;		// Reset the timer0 counter (High byte)
		TMR0L = 0x1F;		// Reset the timer0 counter (Low byte)
		TMR0IF = 0;			// Clear the timer0 interrupt flag
	}
}

// Millisecond delay (accurate to 10mS with a minimum of 10mS)
void delayMs(int milliseconds)
{
	// Timing is performed by timer1
	//
	// Fosc is 48Mhz or 48,000,000 ticks per second, input timing to the timer modules
	// is Fosc/4 or 12,000,000 ticks per second.  There are 1,000 mS per second
	// so 12,000 ticks = 1mS. At 1:8 prescale that's 1500 ticks per mS so we can measure
	// 10mS as 15,000 ticks (65535 - 15000 = 50535 or 0xC567)

	int tensOfMilliseconds;
	int tensLoop;
	
	// Divide the time down into tens of milliseconds and the remaining milliseconds
	tensOfMilliseconds = milliseconds / 10;
	milliseconds = milliseconds - (tensOfMilliseconds * 10);

	// Time the 10 millisecond blocks first
	for (tensLoop = 0; tensLoop < tensOfMilliseconds; tensLoop++)
	{
		TMR1H = 0xC5;
		TMR1L = 0x67;
		TMR1IF = 0; // Reset timer1's flag
		T1CON = 0b10110101;
		while(TMR1IF == 0); // Wait for timer 1 to flag @ 10mS
	}
}

// Initialise the LED states
void initialiseLEDs(unsigned char fadeOnSpeed, unsigned char fadeOffSpeed)
{
	unsigned char ledNumber;

	for (ledNumber = 0; ledNumber < 7; ledNumber++)
	{
		led[ledNumber].fadeOnSpeed = fadeOnSpeed;
		led[ledNumber].fadeOffSpeed = fadeOffSpeed;
	}
}

void fadeAllOn(void)
{
	unsigned char ledNumber;

	for (ledNumber = 0; ledNumber < 7; ledNumber++)
	{
		led[ledNumber].brightness = 63;
		led[ledNumber].fadeOnSpeed = 75;
		led[ledNumber].fadeOffSpeed = 0;
	}

	// Wait for it...
	delayMs(2000);
}

void fadeAllOff(void)
{
	unsigned char ledNumber;

	for (ledNumber = 0; ledNumber < 7; ledNumber++)
	{
		led[ledNumber].brightness = 0;
		led[ledNumber].fadeOnSpeed = 0;
		led[ledNumber].fadeOffSpeed = 40;
	}

	// Wait for it...
	delayMs(1000);
}

void main(void)
{
	// Configure ports as inputs (1) or outputs(0)
	TRISA = 0b00000000;
	TRISB = 0b00000000;
	TRISC = 0b00000000;

	// Clear all ports
	PORTA = 0b00000000;
	PORTB = 0b00000000;
	PORTC = 0b00000000;

	// Disable the USB device
	UCON = 0b00000000;
	UCFG = 0b00001000;

	// Turn off analogue ports
	ADCON1 = 0x0F;

	// Enable interrupts with priority
	IPEN = 1;

	// Set up timer0 to control the PWM
	//
	// We need a minimum cycle speed of 75Hz (to prevent the flicker being noticeable at
	// the lowest brightness).  A frequency of 75Hz is 75 times a second or once every
	// 13,333 microseconds (uS).
	//
	// We want 64 steps of brightness meaning that each 'step' lasts for 13,333 / 64 = 
	// 208uS (i.e. the interrupt must run at 64*75Hz = 4800Hz)
	//
	// Fosc is 48Mhz or 48,000,000 ticks per second, input timing to the timer modules
	// is Fosc/4 or 12,000,000 ticks per second.  There are 1,000,000 uS per second
	// so 12 ticks = 1uS. Therefore 208uS is 12*208 = 2496 ticks at 1:1 prescale value.
	// Minimum prescale is 1:2 so we need an interrupt every 2496/2 = 1248 ticks.
	//
	// If we set timer0 to be 16-bit the maximum counter value is (2^16)-1 = 65535.  Therefore we
	// need to set the initial timer value to 65535 - 1248 = 64287 or 0xFB1F in hexidecimal

	TMR0IP = 0;			// Set timer0 interrupt to low priority
	TMR0IF = 0;			// Clear the timer0 interrupt flag
	TMR0H = 0xFB;		// Reset the timer0 counter (High byte)
	TMR0L = 0x1F;		// Reset the timer0 counter (Low byte)
	T0CON = 0b10000000; // Timer0 on, 16-bit with a 1:2 prescale value
	TMR0IE = 1;			// Enable the timer0 interrupt

	// Before we enable interrupts we must initialise the led states
	int ledNumber;
	for (ledNumber = 0; ledNumber < 7; ledNumber++)
	{
		led[ledNumber].brightness = 0;
		led[ledNumber].fadeOnSpeed = 0;
		led[ledNumber].fadeOffSpeed = 0;
		led[ledNumber].currentBrightness = 0;
		led[ledNumber].fadeCounter = 0;
	}

	// Enable interrupts
	GIEH = 1;			// Global enable all high priority interrupts
	GIEL = 1;			// Global enable all low priority interrupts

	int delayLoop;
	unsigned char startFlag;
	unsigned char displayLoop;

	// Power up delay
	delayMs(1000);

	while(1)
	{
		// 'Knightrider'
		initialiseLEDs(6, 30);
		startFlag = FALSE;
		for (displayLoop = 0; displayLoop < 7; displayLoop++)
		{
			for (ledNumber = 0; ledNumber < 7; ledNumber++)
			{
				led[ledNumber].brightness = 63;
				if (ledNumber == 0) led[6].brightness = 0; else led[ledNumber-1].brightness = 0;
				delayMs(200);

				if (ledNumber == 0 && startFlag == TRUE) delayMs(800); else startFlag = TRUE;
			}
	
			delayMs(800);

			for (ledNumber = 5; ledNumber > 0; ledNumber--)
			{
				led[ledNumber+1].brightness = 0;
				led[ledNumber].brightness = 63;
				delayMs(200);
			}
			led[1].brightness = 0;
		}
		led[0].brightness = 63;
		delayMs(800);
		led[0].brightness = 0;
		delayMs(1000);

		// Pulse all
		initialiseLEDs(6, 30);
		for (displayLoop = 0; displayLoop < 15; displayLoop++)
		{
			// All on
			for (ledNumber = 0; ledNumber < 7; ledNumber++) led[ledNumber].brightness = 63;
			delayMs(800);

			// All off
			for (ledNumber = 0; ledNumber < 7; ledNumber++) led[ledNumber].brightness = 0;
			delayMs(800);
		}

		// Nyssa's police car
		initialiseLEDs(8, 2);
		led[3].brightness = 63;
		for (displayLoop = 0; displayLoop < 60; displayLoop++)
		{
			led[4].brightness = 0;
			led[5].brightness = 0;
			led[6].brightness = 0;
			led[0].brightness = 63;
			led[1].brightness = 63;
			led[2].brightness = 63;
			delayMs(300);

			led[0].brightness = 0;
			led[1].brightness = 0;
			led[2].brightness = 0;
			led[4].brightness = 63;
			led[5].brightness = 63;
			led[6].brightness = 63;
			delayMs(300);
		}
		fadeAllOff();

		// Centre burst
		initialiseLEDs(6, 30);

		for (displayLoop = 0; displayLoop < 18; displayLoop++)
		{
			led[3].brightness = 63;
			delayMs(200);

			led[3].brightness = 0;
			led[2].brightness = 63;
			led[4].brightness = 63;
			delayMs(200);
	
			led[2].brightness = 0;
			led[4].brightness = 0;
			led[1].brightness = 63;
			led[5].brightness = 63;
			delayMs(200);
	
			led[1].brightness = 0;
			led[5].brightness = 0;
			led[0].brightness = 63;
			led[6].brightness = 63;
			delayMs(200);
	
			led[0].brightness = 0;
			led[6].brightness = 0;
			delayMs(800);
		}
		delayMs(800);

		// Sideways shoot L to R
		initialiseLEDs(6, 15);

		for (displayLoop = 0; displayLoop < 12; displayLoop++)
		{
			for (ledNumber = 0; ledNumber < 7; ledNumber++)
			{
				led[ledNumber].brightness = 63;
				delayMs(100);
			}
			for (ledNumber = 0; ledNumber < 7; ledNumber++)
			{
				led[ledNumber].brightness = 0;
			}
			delayMs(300);
			for (ledNumber = 7; ledNumber >= 0; ledNumber--)
			{
				led[ledNumber].brightness = 63;
				delayMs(100);
			}
			for (ledNumber = 7; ledNumber >= 0; ledNumber--)
			{
				led[ledNumber].brightness = 0;
			}
			delayMs(300);
		}

		// Hop-sweep
		initialiseLEDs(3, 15);

		for (displayLoop = 0; displayLoop < 10; displayLoop++)
		{
			led[0].brightness = 63;
			delayMs(200);
			led[2].brightness = 63;
			delayMs(400);
			led[0].brightness = 0;
			led[2].brightness = 0;

			led[1].brightness = 63;
			delayMs(200);
			led[3].brightness = 63;
			delayMs(400);
			led[1].brightness = 0;
			led[3].brightness = 0;

			led[2].brightness = 63;
			delayMs(200);
			led[4].brightness = 63;
			delayMs(400);
			led[2].brightness = 0;
			led[4].brightness = 0;

			led[3].brightness = 63;
			delayMs(200);
			led[5].brightness = 63;
			delayMs(400);
			led[3].brightness = 0;
			led[5].brightness = 0;

			led[4].brightness = 63;
			delayMs(200);
			led[6].brightness = 63;
			delayMs(400);
			led[4].brightness = 0;
			led[6].brightness = 0;
		}

		// Inverse knight-rider
		fadeAllOn();
		initialiseLEDs(10, 10);
		startFlag = FALSE;
		for (displayLoop = 0; displayLoop < 6; displayLoop++)
		{
			for (ledNumber = 0; ledNumber < 7; ledNumber++)
			{
				led[ledNumber].brightness = 0;
				if (ledNumber == 0) led[6].brightness = 63; else led[ledNumber-1].brightness = 63;
				delayMs(400);

				//if (ledNumber == 0 && startFlag == TRUE) delayMs(800); else startFlag = TRUE;
			}

			for (ledNumber = 5; ledNumber > 0; ledNumber--)
			{
				led[ledNumber+1].brightness = 63;
				led[ledNumber].brightness = 0;
				delayMs(400);
			}
			led[1].brightness = 63;
		}
		led[0].brightness = 0;
		delayMs(800);
		led[0].brightness = 63;
		delayMs(1000);

		fadeAllOff();

		// Twinkle-twinkle little LED
		initialiseLEDs(0,0);

		for (displayLoop = 0; displayLoop < 81; displayLoop++)
		{
			led[3].brightness = 63;
			delayMs(30);
			led[3].brightness = 0;
			delayMs(30);

			led[5].brightness = 63;
			delayMs(30);
			led[5].brightness = 0;
			delayMs(30);

			led[1].brightness = 63;
			delayMs(30);
			led[1].brightness = 0;
			delayMs(30);

			led[4].brightness = 63;
			delayMs(30);
			led[4].brightness = 0;
			delayMs(30);

			led[6].brightness = 63;
			delayMs(30);
			led[6].brightness = 0;
			delayMs(30);

			led[2].brightness = 63;
			delayMs(30);
			led[2].brightness = 0;
			delayMs(30);

			led[0].brightness = 63;
			delayMs(30);
			led[0].brightness = 0;
			delayMs(30);
		}


		// Sacha's blinker
		initialiseLEDs(6, 10);
		for (displayLoop = 0; displayLoop < 21; displayLoop++)
		{
			led[1].brightness = 0;
			led[3].brightness = 0;
			led[5].brightness = 0;

			led[0].brightness = 63;
			led[2].brightness = 63;
			led[4].brightness = 63;
			led[6].brightness = 63;
			delayMs(400);

			led[0].brightness = 0;
			led[2].brightness = 0;
			led[4].brightness = 0;
			led[6].brightness = 0;

			led[1].brightness = 63;
			led[3].brightness = 63;
			led[5].brightness = 63;
			delayMs(400);
		}
		fadeAllOff();

		// Edge burst
		initialiseLEDs(6, 30);

		for (displayLoop = 0; displayLoop < 18; displayLoop++)
		{
			led[0].brightness = 63;
			led[6].brightness = 63;
			delayMs(200);

			led[0].brightness = 0;
			led[6].brightness = 0;
			led[1].brightness = 63;
			led[5].brightness = 63;
			delayMs(200);

			led[1].brightness = 0;
			led[5].brightness = 0;
			led[2].brightness = 63;
			led[4].brightness = 63;
			delayMs(200);

			led[2].brightness = 0;
			led[4].brightness = 0;
			led[3].brightness = 63;
			delayMs(800);

			led[3].brightness = 0;
			delayMs(800);
		}
		fadeAllOff();

		// Emma's loopy
		initialiseLEDs(0,4);

		for (displayLoop = 0; displayLoop < 45; displayLoop++)
		{
			for (ledNumber = 0; ledNumber < 7; ledNumber++)
			{
				if (ledNumber > 0) led[ledNumber-1].brightness = 0; else led[6].brightness = 0;
				led[ledNumber].brightness = 63;
				delayMs(100);
			}
		} 
	}
}